package sdsi;

import sdsi.sexp.*;

/**
 * One principal quoting another. The first is actually the principal
 * doing the speaking, the second is only named by the first; there
 * is no intention that the second principal has any awareness or
 * acquiescence to the statement being made.
 *
 * <P>
 * The speaking principal also must explicitly claim to be quoting
 * another; it is a useful mechanism for writing secure multiplexed
 * services. The service is the quoting principal; it quotes those
 * principals on behalf of whom it is performing its service. That
 * way it will not accidentally perform an action for one client
 * using its authority granted to it by another.
 *
 * <P>
 * An extension to SPKI. Relax! Relax! Yes I know extending a security
 * protocol is a dangerous proposition. This extension is justified by
 * my semantic model of SPKI in my thesis. The premise of the model
 * is to give meaning to SPKI's constructs, and this construct preserves
 * the meaning.
 *
 * <P>
 * This thing is a SDSIPrincipal (which really means an Issuer) because
 * we want it to be legal in the issuer field of a certificate. That's
 * sensical in my logic, so it's as valid an extension as any.
 * 
 * @todo ThresholdSubjects should really be ThresholdPrincipals for the
 * same reason.
 *
 * @author jonh@cs.dartmouth.edu
 */
public class Quoting
	extends SDSIPrincipal {

	SDSIPrincipal nameContext;
	Subject quoter;				// the principal doing the quoting
	Subject quotee;				// the principal being quoted

	public static final String LABEL = "quoting";

	/**
	 * @param sexplist An SexpList to parse into a quoting subject.
	 * @param nameContext the principal to whom any names specified in
	 * the sexplist are meaningful.
	 */
	public Quoting(SexpList sexplist, SDSIPrincipal nameContext)
		throws SexpParseException {

		super(sexplist);
		this.nameContext = nameContext;
		if (!sexplist.getType().equals(LABEL)) {
			throw new SexpParseException("Not a quoting subject");
		}
		if (!(sexplist.size() == 3)
			|| !(sexplist.elementAt(1) instanceof SexpList)
			|| !(sexplist.elementAt(2) instanceof SexpList)) {
			throw new
				SexpParseException("Incorrectly formatted quoting subject");
		}
		quoter = (Subject) SDSIObject.principalParse(
					(SexpList) sexplist.elementAt(1), nameContext);
		quotee = (Subject) SDSIObject.principalParse(
					(SexpList) sexplist.elementAt(2), nameContext);
	}

	public Quoting(Subject quoter, Subject quotee)
		throws SexpParseException {

		this.quoter = quoter;
		this.quotee = quotee;
		SexpList sexplist = (new SexpList())
			.append(LABEL)
			.append(quoter.getSrep())
			.append(quotee.getSrep());
		srep = sexplist;
	}

	public Subject getQuoter() { return quoter; }
	public Subject getQuotee() { return quotee; }

	public String toShortString() {
		return quoter.toShortString()+" quoting "+quotee.toShortString();
	}
}	
