package jp;

/**
 * The definition of constants in the Snowflake version of the HTTP
 * Authorization protocol.
 */
public interface SfHttpProtocol {
	/**
	 * Sent in an HTTP response (how's that for confusing? It's
	 * because the server is challenging the client) to demand
	 * HTTP authorization of the client.
	 */
	public static final String HTTP_AUTH_CHALLENGE = "WWW-Authenticate";

	/**
	 * The client responds to the HTTP_AUTH_CHALLENGE in its second
	 * request, transmitting its proof of authority in the value of this
	 * header.
	 */
	public static final String HTTP_AUTH_RESPONSE = "Authorization";

	/**
	 * The identifier for the Snowflake authorization method, supplied
	 * as the first word after the HTTP_AUTH_CHALLENGE.
	 */
	public static final String SNOWFLAKEPROOF = "SnowflakeProof ";

	/**
	 * Means the client needs to authorize itself; service issuer and
	 * minimum tag are supplied as arguments in {@link #SERVICEISSUER}
	 * and {@link #MINIMUMTAG} headers.
	 */
	public static final String AUTHORIZECLIENT = "Authorize-Client";

	/**
	 * A challenge that tells the client to name a principal it wishes
	 * to use as its identity. (The client doesn't have to prove
	 * anything about that principal yet.) This challenge is presented
	 * by a gateway trying to learn on whose behalf it should operate.
	 */
	public static final String IDENTIFYCLIENT = "Identify-Client";

	/**
	 * Means the client needs to give the proxy (server in this
	 * transaction) authority to perform the transaction. Ultimate
	 * service issuer, minimum tag, and proxy principal who will be
	 * quoting the issuer are supplied.<p>
	 * 
	 * A client that blindly responds to this challenge without
	 * considering the trustworthiness of the challenging gateway risks
	 * falling for a man-in-the-middle attack. The client should also
	 * consider the strength of the delegation requested to avoid
	 * placing too much trust in the gateway.<p>
	 *
	 * Note that ``challenge'' is not the best word for this demand; the
	 * gateway (or ``proxy,'' as I sometimes refer to the gateway in this
	 * code) is actually requesting that the client delegate some authority
	 * to it.
	 */
	public static final String AUTHORIZEPROXY = "Authorize-Proxy";

	/**
	 * An extra header that carries information about the demands of an
	 * {@link #HTTP_AUTH_CHALLENGE}. This header specifies the issuer that
	 * the proof must show the request speaks for.
	 */
	public static final String SERVICEISSUER =	"Sf-ServiceIssuer";

	/**
	 * An extra header that carries information about the demands of an
	 * {@link #HTTP_AUTH_CHALLENGE}. This header specifies the minimum
	 * restriction set that contains the request that inspired the challenge.
	 */
	public static final String MINIMUMTAG =		"Sf-MinimumTag";

	/**
	 * An extra header that describes the principal that must speak for
	 * the client (and eventually the resource server); presented with
	 * AUTHORIZEPROXY challenges.
	 */
	public static final String PROXYPRINCIPAL =	"Sf-ProxyPrincipal";

	/**
	 * An extra header, now unused. Once designed to carry extra facts
	 * either direction to be deposited in the recipient's Prover.
	 * 
	 * @deprecated now unused.
	 */
	public static final String AUXILLIARYFACTS ="Sf-AuxilliaryFacts";

	/**
	 * An extra header indicating the client's identity in response to
	 * an {@link #IDENTIFYCLIENT} challenge.
	 */
	public static final String CLIENTIDENTITY =	"Sf-ClientIdentity";

	/**
	 * The client sends this extra header, with an argument giving its
	 * public key, to request that the server generate a secret MAC
	 * (Message Authentication Code) and return it to the client
	 * encrypted with the client's public key.<p>
	 * 
	 * The server should be careful to ensure that any REQUESTMAC
	 * header belongs to the signed text of a request, since it assumes
	 * a delegation from the MAC itself to the signer of the request.
	 * An adversary could otherwise inject a request for a MAC into a
	 * message, and steal the client's authority.<p>
	 * 
	 * The MAC protocol is something I brewed up, and it depends on
	 * secrecy, which my logic says little about. It would be prudent
	 * to study this protocol further or substitute a better-known
	 * protocol before trying to deploy this protocol in production.
	 */
	public static final String REQUESTMAC 	=	"Sf-RequestMac";

	/**
	 * The server sends the encrypted MAC back to the client in this
	 * header.
	 */
	public static final String ENCRYPTEDMAC =	"Sf-EncryptedMac";

	/**
	 * A server sends to the client in this extra header a proof that
	 * the document content (bytes following the headers and first
	 * blank line) of this message speak for a symbolic name bound in
	 * the server's secure SPKI namespace. The document is represented
	 * in the proof by its SPKI {@link sdsi.ObjectHash}. This a simple
	 * form of server authentication; note that it does <em>not</em>
	 * verify the authority of the headers returned by the server, so
	 * (for example) it does not protect the MAC protocol from MITM
	 * attacks.
	 */
	public static final String DOCFORSERVERNAME =	"Sf-DocSpeaksForServerName";
}
