/*
 * (c) Copyright IBM Corp. 2005 All Rights Reserved
 *
 * Physical Memory Information Module
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 * @file physmem.h
 *
 * @brief Definitions for the physical memory module.
 *
 * This part provides the source for the literal values, global variables,
 * and structures for the physmem_info module.
 *
 * @author International Business Machines
 * @author Paul Movall <movall@us.ibm.com>
 *
 * @version Current Version: 1.0
 *
 * @date Current Date: 01/2005
 *
 * @version 0.1, 11/2003: File created by Paul Movall <movall@us.ibm.com>
 * @version 1.0, 01/2005: Miscellaneous cleanup for publish
 *
 */

/**
 * @mainpage
 *
 * This module provides for a mechanism to extract physical memory
 * usage for kernel and user space processes running the Linux
 * kernel.
 *
 * @section Rationale
 * In most Linux environments, the standard memory metric is virtual memory.
 * This metric can be used to determine the swap space needed as well
 * as swap rates and other metrics related to swapping. This can be seen
 * by the prevalence of various virtual memory tools, such as mpatrol or
 * even the /proc/@<pid@>/maps node.
 *
 * However in an environment without a swap device, the virtual
 * memory metric is not as important as physical memory. In such a system,
 * physical memory usage is a critical metric. Virtual memory tools as
 * noted above are still needed, but they are of limited use when
 * the system has exhausted all of its physical memory. When this
 * happens, the kernel will arbitrarily start killing processes or
 * just panic. This is not a desirable situation.
 *
 * Therefore the understanding of the assignment and usage patterns of
 * physical pages is vital. During a review of existing open source projects,
 * no viable tool to perform such analysis was found. This tool set,
 * consisting of a dynamically loadable kernel module and
 * a post-processing analyzer were designed and implemented to perform
 * such analysis.
 *
 * This is the kernel module portion of the toolset.
 */

#ifndef _PHYSMEM_H
#define _PHYSMEM_H

/** @def PHYSMEM_IOC_MAGIC ioctl command magic value definition. */
#define PHYSMEM_IOC_MAGIC		'p'

/** @def PHYSMEM_IOCTL_MIN_SIZE Query minimum read size. */
#define PHYSMEM_IOCTL_MIN_SIZE		_IO(PHYSMEM_IOC_MAGIC,  0)

/** @def PHYSMEM_IOCTL_CUR_PID Query current task's pid. */
#define PHYSMEM_IOCTL_CUR_PID		_IO(PHYSMEM_IOC_MAGIC, 1)

/** @def PHYSMEM_IOC_MAX Maximum value of ioctl command value. */
#define PHYSMEM_IOC_MAX			1

/** @def PROCFILE_PHYSMEM_DIR Name of directory in /proc. */
#define PROCFILE_PHYSMEM_DIR "physical_info"

/** @def PROCFILE_MEM_TASK_DEBUG Name of task info node in /proc. */
#define PROCFILE_MEM_TASK_DEBUG "debug_task_mem"

#ifdef __KERNEL__
#ifndef PG_cache
/** @def PG_cache Value of PG_ flags that signify this is a cache page. */
#define PG_cache 0
#endif

#ifndef TRUE
/** @def TRUE Logical value of true. */
#define TRUE 1
#endif

#ifndef FALSE
/** @def FALSE Logical value of false. */
#define FALSE 0
#endif

/* ========================================================================== */
/**
 * @struct mem_task_info_struct
 *
 * @brief Private data per open for the device representing task memory usage.
 *
 * This data structure is set as the private data in the open file handle.
 * It is used to validate the requests, signify when we have reached the
 * end of the task list, and to store the minimum size of the read
 * operation. Note that the task list is really a circular list (i.e.,
 * it is not NULL terminated), so care must be taken to insure that
 * we have reached the end of the list.
 */
/**
 * @typedef mem_task_info_t
 * @brief Type definition for the mem_task_info_struct.
 */
typedef struct mem_task_info_struct {
	unsigned long valid; /**< This field is used for request validation. */
	int min_size;	/**< This field is used to hold the maximum size
			 * that a read may return, so it should be the
			 * minimum size of the buffer on a read request. */
	int done;	/**< This field is used to signify the end of the
			 * task list. */
} mem_task_info_t;
/** @def MEM_TASK_STRUCT_VALID Used as the validation flag in the structure. */
#define MEM_TASK_STRUCT_VALID	0xCAFEBABE

/**
 * @var procfile_physmem_dir
 * @brief Handle for the /proc directory of this module.
 */
struct proc_dir_entry *procfile_physmem_dir;

#endif

#endif /* _PHYSMEM_H */

