%{
/*
 * (c) Copyright IBM Corp. 2005 All Rights Reserved
 *
 * Physical Memory Information Module
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * at your option any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/**
 * @file td.flex
 *
 * @brief Input data parser.
 *
 * This part provides the source for parsing of the input data.
 * The input data is specified by the kernel module or the crash utility.
 * This parser reads the data into internal data structures for further
 * analysis.
 *
 * There is one piece of analysis done here. That is the determination
 * if a VMA is a bss segment. This is determined by the convention of
 * the VMA is writable, it is not associated with a file, the previous
 * VMA is contiguous and writable but is associated with a file.
 *
 * @author International Business Machines
 * @author Paul Movall <movall@us.ibm.com>
 *
 * @version Current Version: 1.0
 *
 * @date Current Date: 01/2005
 *
 * @version 0.1, 11/2003: File created by Paul Movall <movall@us.ibm.com>
 * @version 1.0, 01/2005: Miscellaneous cleanup for publish
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include "td.h"

#define ECHO
#define SEGMENT_MASK 0xF0000000
task_t *cur_task;
vma_t *cur_vma;
pte_t *cur_pte;
map_t *cur_map;
int stack_segment = -1;

%}

ach [a-zA-Z0-9\&\*\@\:\$\>\(\<\-\^\#\%\+\?\_\"\)\!\~\/]
numeral [0-9]
space [ \t]
tab [\t]
new   [ \t\n\r]
comma [\,]
equal [\=]
pipe [\|]

%Start tname tpid tmm tproc tminf tmajf tcminf tcmajf tutime tstime nextvma vstart vend vvsize vrsize vpflags vvflags vname nextpte pusr pexec prd psingle pshared nextmap nexttask midmap mphys mcount mflags

%%

<INITIAL,nextvma,nextmap,midmap,nexttask,tproc,tminf>^(T:){space} {
	if (!task_valid) {
		/* the previous task was not used so free the alloc'ed data */
		free(cur_task);
	} else {
		/* previous task was used, can't free it since it is now in the
		 * list
		 */
	}
	cur_task = (task_t *)malloc(sizeof(task_t));
	if (!cur_task) {
		perror("Error allocating task memory: ");
		exit(-1);
	}
	memset(cur_task, 0, sizeof(task_t));
	cur_task->next = NULL;
	cur_task->vma_head = NULL;
	cur_task->vma_tail = NULL;
	task_valid = FALSE;
	BEGIN tname;
#ifdef LEX_DEBUG
	fprintf(stderr, "Found task - ");
#endif
}

<tname>{ach}+ {
	strcpy(cur_task->name, yytext);
	BEGIN tpid;
#ifdef LEX_DEBUG
	fprintf(stderr, "name: %s, ", yytext);
#endif
}

<tpid>{numeral}+ {
	BEGIN tmm;
#ifdef LEX_DEBUG
	fprintf(stderr, "pid: %s, ", yytext);
#endif
}

<tmm>{ach}+ {
	cur_task->mm = xtoi(yytext);
	/* now that have the mm, check to see if that mm is already in
	 * another task */
	if (!find_task(cur_task)) {
		/* it was not, so add it to the list... */
		add_task(cur_task);
		/* ... and mark it as valid */
		task_valid = TRUE;
	} else {
		/* already have vm detail so mark it to skip */
		task_valid = FALSE;
	}

	BEGIN tproc;
#ifdef LEX_DEBUG
	fprintf(stderr, "mm: %08lX [%d]\n ",
		cur_task->mm, task_valid);
#endif
}

<tproc>{numeral}+ {
	BEGIN tminf;
}

<tminf>{numeral}+ {
	if (task_valid) {
		cur_task->minor_flts = atoi(yytext);
	}
	BEGIN tmajf;
#if defined(LEX_DEBUG) || defined(LEX_TASK_DEBUG)
	fprintf(stderr, "minor faults: %08lu, ", cur_task->minor_flts);
#endif
}

<tmajf>{numeral}+ {
	if (task_valid) {
		cur_task->major_flts = atoi(yytext);
	}
	BEGIN tcminf;
#if defined(LEX_DEBUG) || defined(LEX_TASK_DEBUG)
	fprintf(stderr, "major faults: %08lu, ", cur_task->major_flts);
#endif
}

<tcminf>{numeral}+ {
	if (task_valid) {
		cur_task->chld_min_flts = atoi(yytext);
	} else {
		cur_task->chld_min_flts += atoi(yytext);
	}
	BEGIN tcmajf;
#if defined(LEX_DEBUG) || defined(LEX_TASK_DEBUG)
	fprintf(stderr, "children minor faults: %08lu, ",
		cur_task->chld_min_flts);
#endif
}

<tcmajf>{numeral}+ {
	if (task_valid) {
		cur_task->chld_maj_flts = atoi(yytext);
	} else {
		cur_task->chld_maj_flts += atoi(yytext);
	}
	BEGIN tutime;
#if defined(LEX_DEBUG) || defined(LEX_TASK_DEBUG)
	fprintf(stderr, "children major faults: %08lu, ",
		cur_task->chld_maj_flts);
#endif
}

<tutime>{numeral}+ {
	if (task_valid) {
		cur_task->user_time = atoi(yytext);
	} else {
		cur_task->user_time += atoi(yytext);
	}
	BEGIN tstime;
#if defined(LEX_DEBUG) || defined(LEX_TASK_DEBUG)
	fprintf(stderr, "user time: %08lu, ",
		cur_task->user_time);
#endif
}

<tstime>{numeral}+ {
	if (task_valid) {
		cur_task->sys_time = atoi(yytext);
	} else {
		cur_task->sys_time += atoi(yytext);
	}
	BEGIN nextvma;
#if defined(LEX_DEBUG) || defined(LEX_TASK_DEBUG)
	fprintf(stderr, "system time: %08lu, ",
		cur_task->sys_time);
#endif
}

<tproc,tminf,nextvma,nextmap,midmap>^(V:){space} {
	if (task_valid) {
		cur_vma = (vma_t *)malloc(sizeof(vma_t));
		if (!cur_vma) {
			perror("Error allocating vma memory: ");
			exit(-1);
		}
		memset(cur_vma, 0, sizeof(vma_t));
		cur_vma->pte_info = NULL;
		cur_vma->map_head = NULL;
		cur_vma->map_tail = NULL;
		cur_vma->next = NULL;
		cur_vma->prev = NULL;
		add_vma(cur_task, cur_vma);
	}
	BEGIN vstart;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "Found VMA - ");
#endif
}

<vstart>{ach}+ {
	if (task_valid) {
		cur_vma->start = xtoi(yytext);
	}
	BEGIN vend;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "start: %08lX, ", cur_vma->start);
#endif
}

<vend>{ach}+ {
	if (task_valid) {
		cur_vma->end = xtoi(yytext);
	}
	BEGIN vvsize;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "end: %s, ", yytext);
#endif
}

<vvsize>{ach}+ {
	if (task_valid) {
		cur_vma->vlen = xtoi(yytext);
	}
	BEGIN vrsize;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "virtual size: %s, ", yytext);
#endif
}

<vrsize>{ach}+ {
	if (task_valid) {
		cur_vma->rlen = xtoi(yytext);
	}
	BEGIN vpflags;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "resident size: %s, ", yytext);
#endif
}

<vpflags>{ach}+ {
	if (task_valid) {
		cur_vma->pte_flags = xtoi(yytext);
	}
	BEGIN vvflags;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "pte flags: %s, ", yytext);
#endif
}

<vvflags>{ach}+ {
	if (task_valid) {
		cur_vma->vma_flags = xtoi(yytext);
	}
	BEGIN vname;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "vma flags: %s, ", yytext);
#endif
}

<vname>{ach}* {
	if (task_valid) {
		strcpy(cur_vma->name, yytext);
		strcpy(cur_vma->assoc_name, "None");

		if ( ((cur_vma->vma_flags & VMA_FLAGS_EXECUTABLE) ==
		      VMA_FLAGS_EXECUTABLE) &&
		     ((cur_vma->vma_flags & VMA_FLAGS_MASK_RWX) ==
		      VMA_FLAGS_RWX) &&
		     (cur_vma->prev) ) {
			/*
			 * hack for having executables with rwx vma
			 * i.e., built with -fpic
			 */
			if ((cur_vma->prev->vma_flags &
			     VMA_FLAGS_EXECUTABLE) == VMA_FLAGS_EXECUTABLE) {
				/*
				 * the .text vma comes first immediately
				 * followed by the .data vma,
				 * with both marked for the executable.
				 * If the current vma is from the
				 * executable and the previous vma is also
				 * from the executable, then the previous
				 * one is the .text so clear the write
				 * flag (flags to r.x instead of rwx)
				 */
				cur_vma->prev->vma_flags &= ~VMA_FLAGS_WR;
			}
		}

		if ( (cur_vma->prev) &&
		     (strcmp(cur_vma->name, "NULL") == 0) &&
		     (strcmp(cur_vma->prev->name, "NULL") != 0) &&
		     ((cur_vma->vma_flags & VMA_FLAGS_WR) == VMA_FLAGS_WR) &&
		     ((cur_vma->prev->vma_flags & VMA_FLAGS_WR) ==
		      VMA_FLAGS_WR) &&
		     (cur_vma->start == (cur_vma->prev->end + 1)) ) {
			/*
			 * this vma is anonymous,
			 * the previous vma is not anonymous,
			 * both are writable,
			 * they are adjacent,
			 * so must be .bss
			 */
			strcpy(cur_vma->assoc_name,
			       cur_vma->prev->name);
			cur_vma->vma_flags |= VMA_FLAGS_BSS;

			if ((cur_vma->prev->vma_flags &
			     VMA_FLAGS_EXECUTABLE) == VMA_FLAGS_EXECUTABLE) {
				/* if the previous is executable set flag here*/
				cur_vma->vma_flags |= VMA_FLAGS_EXECUTABLE;
			}
		}

		if (cur_vma->vma_flags & VMA_FLAGS_STACK) {
			/* also set associated name for stack area */
			strcpy(cur_vma->assoc_name,
			       cur_task->name);

			if (stack_segment == -1) {
				stack_segment = cur_vma->start & SEGMENT_MASK;
			}
		}

		if ( (cur_vma->prev) &&
		     (strcmp(cur_vma->name, "NULL") == 0) &&
		     (strcmp(cur_vma->prev->name, "NULL") == 0) &&
		     ((cur_vma->vma_flags & VMA_FLAGS_WR) == VMA_FLAGS_WR) &&
		     (pstack_filter) &&
		     ((cur_vma->vlen / PAGE_SIZE) == 511) &&
		     ((cur_vma->start & SEGMENT_MASK) == stack_segment) ) {
			/* reduce pthread stack to something more meaningful */
			cur_vma->vlen = pstack_size * PAGE_SIZE;
			cur_vma->end = cur_vma->start + cur_vma->vlen;
			fprintf(stderr, "Reducing pthread stack for %s, new size %d pages\n",
				cur_task->name,
				(cur_vma->end - cur_vma->start) / PAGE_SIZE);
		}
	}

	BEGIN nextpte;
#if defined(LEX_DEBUG) || defined(LEX_VMA_DEBUG)
	fprintf(stderr, "name: %s\n", yytext);
#endif
}

<vname,nextpte>^({tab})*(P:){space} {
	if (task_valid) {
		cur_pte = (pte_t *)malloc(sizeof(pte_t));
		if (!cur_pte) {
			perror("Error allocating pte memory: ");
			exit(-1);
		}
		memset(cur_pte, 0, sizeof(pte_t));
		cur_vma->pte_info = cur_pte;
	}
	BEGIN pusr;
#ifdef LEX_DEBUG
	fprintf(stderr, "Found PTE - ");
#endif
}

<pusr>{ach}+ {
	if (task_valid) {
		cur_pte->user_space_count = atoi(yytext);
	}
	BEGIN pexec;
#ifdef LEX_DEBUG
	fprintf(stderr, "usr: %s, ", yytext);
#endif
}

<pexec>{ach}+ {
	if (task_valid) {
		cur_pte->exec_count = atoi(yytext);
	}
	BEGIN prd;
#ifdef LEX_DEBUG
	fprintf(stderr, "exec: %s, ", yytext);
#endif
}

<prd>{ach}+ {
	if (task_valid) {
		cur_pte->rd_only_count = atoi(yytext);
	}
	BEGIN psingle;
#ifdef LEX_DEBUG
	fprintf(stderr, "rd only: %s, ", yytext);
#endif
}

<psingle>{ach}+ {
	if (task_valid) {
		cur_pte->single_use_count = atoi(yytext);
	}
	BEGIN pshared;
#ifdef LEX_DEBUG
	fprintf(stderr, "single use pages: %s, ", yytext);
#endif
}

<pshared>{ach}+ {
	if (task_valid) {
		cur_pte->shared_count = atoi(yytext);
	}
	BEGIN nextmap;
#ifdef LEX_DEBUG
	fprintf(stderr, "shared pages: %s\n", yytext);
#endif
}

<nextmap>^(M:){space} {
	BEGIN midmap;
#ifdef LEX_DEBUG
	fprintf(stderr, "Found Map, skipping...\n");
#endif
}

<midmap>{ach}+{equal} {
	if (task_valid) {
		cur_map = (map_t *)malloc(sizeof(map_t));
		if (!cur_map) {
			perror("Error allocating map memory: ");
			exit(-1);
		}
		memset(cur_map, 0, sizeof(map_t));
		cur_map->next = NULL;
		cur_map->virt = xtoi(yytext);

		if (cur_vma->map_tail == NULL) {
			cur_vma->map_tail = cur_map;
			cur_vma->map_head = cur_map;
		} else {
			cur_vma->map_tail->next = cur_map;
			cur_vma->map_tail = cur_map;
		}
	}
	BEGIN mphys;
}

<mphys>{ach}+{comma} {
	if (task_valid) {
		cur_map->phys = xtoi(yytext);
	}
	BEGIN mcount;
}

<mcount>{ach}+{comma} {
	if (task_valid) {
		cur_map->count = xtoi(yytext);
	}
	BEGIN mflags;
}

<mflags>{ach}+{space}*{pipe} {
	if (task_valid) {
		cur_map->flags = xtoi(yytext);
	}
	BEGIN midmap;
}

%%

yywrap()
{
    return(1);
}

